import os
import yaml

class project:
    # 初始化配置
    def __init__(self, configPath: str):
        # 确保配置路径是相对于 config 目录的
        # self.configPath = os.path.join(os.path.dirname(__file__), configPath)
        self.configPath = configPath
        self.config = {}
        self._loadConfigs()
    
    # 加载指定目录下的所有 YAML 配置文件
    def _loadConfigs(self):
        self.config.clear()

        # name 和 path
        self.name = self.configPath.split('/')[0]
        self.path = os.path.join(os.path.dirname(__file__), self.configPath)
        self.config['name'] = self.name
        self.config['path'] = self.path

        if not os.path.exists(self.path):
            raise FileNotFoundError(f"配置目录不存在: {self.configPath}")
            
        for fileName in os.listdir(self.path):
            if fileName.endswith('.yaml') or fileName.endswith('.yml'):
                filePath = os.path.join(self.path, fileName)
                configName = os.path.splitext(fileName)[0]
                
                with open(filePath, 'r', encoding='utf-8') as f:
                    self.config[configName] = yaml.safe_load(f)

    # 更新配置
    def readConfig(self, configPath: str):
        self.configPath = configPath
        self._loadConfigs()
        from lib.common import db
        db.initClass()  # 初始化数据库类

    # 获取指定配置文件的内容，支持通过.分层访问
    # now 为当前配置，如果存在，则返回当前配置
    def getConfig(self, key: str, now=None):
        # 如果传入了 now，则直接返回 now
        if now is not None:
            return now

        # 如果key中间含有.，则进行分层访问
        keys = key.split('.')
        value = self.config
        
        # 通过keys逐层访问配置
        for k in keys:
            if isinstance(value, dict):
                value = value.get(k)
                # 如果往下访问的值是None，则返回None
                if value is None:
                    return None
            else:
                return None
        
        return value
    
    def getConfigValue(self, file, key, now=None):
        return self.getConfig(f'{file}.{key}', now)

    # 判断配置文件中是否存在para，para是config字段的key，如果para是多层的，则用.分隔，如果para中有|分隔符，则表示多个key中要全部存在
    def has(self, para):
        keyList = [s for s in para.split('.') if s]
        current = self.config
        for arg in keyList:
            keys = [s for s in arg.split('|') if s]
            if not all(key in current for key in keys):
                return False
            current = current[keys[0]]
        return True

    # 获取所有配置文件的内容
    def getAllConfigs(self):
        return self.config

    # 通过config的信息更新info里面的信息，然后返回info
    def update(self, info, configFile):
        config = self.getConfig(configFile)
        for key, value in info.items():
            if key in config and not info[key]:
                info[key] = config[key]

            if isinstance(value, dict):
                for k, v in value.items():
                    if k in config:
                        value[k] = config[k]

        return info

    # 支持点号分层访问配置文件
    def __getattr__(self, key: str):
        return self.getConfig(key)

    # 获取指定配置文件的内容
    def __getitem__(self, key: str):
        return self.config[key]
    
    # 检查指定配置文件是否存在
    def __contains__(self, key: str):
        return key in self.config
    
    # 迭代所有配置文件
    def __iter__(self):
        return iter(self.config)

PROJECT = project('project_name/cloud_name/env_name')

# 使用示例
if __name__ == '__main__':
    # 更新配置文件
    PROJECT.readConfig('project_name/cloud_name/env_name')

    # 名字
    print("项目名称:", PROJECT.name)
    # 获取特定配置，支持点号分层访问
    dbConfig = PROJECT.getConfig('db.deploy_db.host')
    print("数据库配置:", dbConfig)
    
    # 获取完整配置
    aliyunConfig = PROJECT.getConfig('aliyun')  # 获取完整的 aliyun.yaml 内容
    print("阿里云配置:", aliyunConfig)
    
    # 获取所有配置
    allConfigs = PROJECT.getAllConfigs()
    print("所有配置:", allConfigs) 