import boto3
import re
import logging
import os
import shutil
import zipfile
import tempfile
import json
import time
import sys

# Configure logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class AuroraSlowLogManager:
    DEFAULT_LAMBDA_FUNCTION_NAME = 'aurora-slowlog-processor'
    DEFAULT_LAMBDA_ROLE_NAME = 'aurora-slowlog-processor-role'
    
    def __init__(self, credential):
        """
        初始化Aurora Slowlog管理器
        Args:
            credential: AWS认证信息
        """
        self.rds = boto3.client('rds', **credential)
        self.cloudwatch = boto3.client('logs', **credential)
        self.lambda_client = boto3.client('lambda', **credential)
        self.iam = boto3.client('iam', **credential)
        self.sts = boto3.client('sts', **credential)
        self.credential = credential
        self.region = credential.get('region_name', 'us-east-1')
        self.account_id = None
        logger.info("AuroraSlowLogManager initialized")
    
    def _check_and_create_slowlog(self, cluster):
        """
        根据模式检查并配置Aurora慢查询日志
        Args:
            cluster: Aurora集群信息
        """
        # 检查慢查询日志配置
        config_status = self._check_slowlog_config(cluster)
        # 如果未配置或配置错误，进行自动配置
        if not config_status['slow_query_log'] or not config_status['log_queries_not_using_indexes']:
            logger.info(f"Slowlog not configured for cluster {cluster['DBClusterIdentifier']}")
        else:
            self._configure_slowlog_export(cluster)
            # 不再直接创建指标筛选器，而是通过Lambda处理器实现
            # 注意：Lambda订阅将在deploy_lambda方法中统一处理
            logger.info(f"Slowlog configured for cluster {cluster['DBClusterIdentifier']}")
            
            lambda_arn=self.deploy_lambda()
            logger.info("DB SlowLog Processor Lambda function deployed successfully")
            self.create_log_subscription(cluster, lambda_arn)
            
            logger.info(f"Slowlog Lambda deployed for cluster {cluster['DBClusterIdentifier']}")
            
    def deploy_lambda(self, function_name=DEFAULT_LAMBDA_FUNCTION_NAME, role_name=DEFAULT_LAMBDA_ROLE_NAME):
        """
        部署Aurora慢查询处理Lambda函数
        Args:
            function_name: Lambda函数名称，默认为DEFAULT_LAMBDA_FUNCTION_NAME
            role_name: Lambda执行角色名称，默认为DEFAULT_LAMBDA_ROLE_NAME
        Returns:
            str: Lambda函数ARN
        """
        try:
            response = self.lambda_client.get_function(FunctionName=function_name)
            logger.info(f"Lambda function {function_name} already exists")
            return response['Configuration']['FunctionArn']
        except self.lambda_client.exceptions.ResourceNotFoundException:
            pass
        
        try:    
            function_name = function_name or self.DEFAULT_LAMBDA_FUNCTION_NAME
            role_name = role_name or self.DEFAULT_LAMBDA_ROLE_NAME
             # 确保Lambda函数已经准备好接收调用
          
            # 创建Lambda执行角色
            role_arn = self.create_lambda_role(role_name)
            logger.info(f"Lambda execution role ARN: {role_arn}")
            logger.info(f"Waiting for role to be fully ready...")
            time.sleep(15)
            # 创建Lambda部署包
            zip_content = self.create_lambda_package()
            logger.info("Created Lambda deployment package")
            
            # 创建或更新Lambda函数
            lambda_arn = self.create_or_update_lambda(function_name, role_arn, zip_content)
            logger.info(f"Lambda function ARN: {lambda_arn}")
            return lambda_arn
        except Exception as e:
            logger.error(f"Error deploying Lambda function: {str(e)}")
            raise

    #检查并配置Aurora慢查询日志
    def check_and_create_slowlog_by_pattern(self, pattern):
        self._scan_aurora_clusters_by_pattern(self._check_and_create_slowlog, pattern)

    def _check_slowlog_config(self, cluster):
        """
        检查Aurora集群的慢查询日志是否启用并正确配置
        Args:
            cluster: Aurora集群信息
        Returns:
            dict: 配置状态
        """
        try:
            # 初始化默认配置
            slowlog_config = {
                'slow_query_log': None,
                'long_query_time': None,
                'log_output': None,
                'min_examined_row_limit': None,
                'log_queries_not_using_indexes': None
            }
            
            # 获取参数组名称
            parameter_group_name = cluster.get('DBClusterParameterGroup')
            
            # 如果找到参数组名称，获取参数组设置
            if parameter_group_name:
                logger.info(f"Using parameter group: {parameter_group_name}")
                paginator = self.rds.get_paginator('describe_db_cluster_parameters')
                params = {'Parameters': []}
                for page in paginator.paginate(DBClusterParameterGroupName=parameter_group_name):
                    params['Parameters'].extend(page['Parameters'])
                for param in params['Parameters']:
                    if param['ParameterName'] in slowlog_config:
                        logger.info(f"Using ParameterName: {param['ParameterName']}")
                        if 'ParameterValue' in param:
                            logger.info(f"Using ParameterName: {param['ParameterName']} ParameterValue: {param['ParameterValue']}")
                            slowlog_config[param['ParameterName']] = param['ParameterValue']
                return slowlog_config     
        except Exception as e:
            
            logger.error(f"Error checking slowlog config: {str(e)}")
            # 返回默认配置而不是抛出异常
            return {
                'slow_query_log': None,
                'long_query_time': None,
                'log_output': None,
                'log_queries_not_using_indexes': None,
                'min_examined_row_limit': None
            }

    def _get_account_id(self):
        """
        获取当前AWS账户ID
        Returns:
            str: AWS账户ID
        """
        if not self.account_id:
            try:
                self.account_id = self.sts.get_caller_identity().get('Account')
            except Exception as e:
                logger.error(f"Error getting AWS account ID: {str(e)}")
                raise
        return self.account_id

    def _configure_slowlog_export(self, cluster):
        """
        配置Aurora慢查询日志导出到CloudWatch LogGroup
        Args:
            cluster: Aurora集群信息
            log_group_name: CloudWatch LogGroup名称
        """
        try:
            # 检查是否为集群对象
            if 'DBClusterIdentifier' in cluster:
                # 这是一个Aurora集群对象
                cluster_id = cluster['DBClusterIdentifier']
                logger.info(f"Configuring slowlog export for db cluster: {cluster_id}")
                
                # 启用慢查询日志导出到CloudWatch
                # Get current export configuration
                response = self.rds.describe_db_clusters(DBClusterIdentifier=cluster_id)
                current_export_types = response['DBClusters'][0].get('EnabledCloudwatchLogsExports', [])
                
                # Only modify if slowquery is not already enabled
                if 'slowquery' not in current_export_types:
                    current_export_types.extend(['slowquery'])
                    self.rds.modify_db_cluster(
                        DBClusterIdentifier=cluster_id,
                        CloudwatchLogsExportConfiguration={
                            'EnableLogTypes': current_export_types,
                            'DisableLogTypes': []
                        },
                        ApplyImmediately=True
                    )
                
                logger.info(f"Configured slowlog export for db cluster {cluster_id}")
        except Exception as e:
            logger.error(f"Error configuring slowlog export: {str(e)}")

    
        
    def create_lambda_role(self, role_name=None):
        """
        创建Lambda执行角色
        Args:
            role_name: Lambda执行角色名称，默认为DEFAULT_LAMBDA_ROLE_NAME
        Returns:
            str: 角色ARN
        """
        role_name = role_name or self.DEFAULT_LAMBDA_ROLE_NAME
        try:
            # 检查角色是否已存在
            try:
                response = self.iam.get_role(RoleName=role_name)
                logger.info(f"Role {role_name} already exists")
                return response['Role']['Arn']
            except self.iam.exceptions.NoSuchEntityException:
                pass
            
            # 创建角色
            assume_role_policy = {
                "Version": "2012-10-17",
                "Statement": [
                    {
                        "Effect": "Allow",
                        "Principal": {"Service": "lambda.amazonaws.com"},
                        "Action": "sts:AssumeRole"
                    }
                ]
            }
            
            response = self.iam.create_role(
                RoleName=role_name,
                AssumeRolePolicyDocument=json.dumps(assume_role_policy),
                Description="Role for Aurora SlowLog Processor Lambda function"
            )
            
            # 附加策略
            self.iam.attach_role_policy(
                RoleName=role_name,
                PolicyArn="arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
            )
            
            # 创建自定义策略允许写入CloudWatch指标
            policy_name = f"{role_name}-cloudwatch-policy"
            policy_document = {
                "Version": "2012-10-17",
                "Statement": [
                    {
                        "Effect": "Allow",
                        "Action": [
                            "cloudwatch:PutMetricData"
                        ],
                        "Resource": "*"
                    }
                ]
            }
            
            try:
                policy_response = self.iam.create_policy(
                    PolicyName=policy_name,
                    PolicyDocument=json.dumps(policy_document),
                    Description="Allow Aurora SlowLog Processor to write CloudWatch metrics"
                )
                policy_arn = policy_response['Policy']['Arn']
            except self.iam.exceptions.EntityAlreadyExistsException:
                account_id = self._get_account_id()
                policy_arn = f"arn:aws:iam::{account_id}:policy/{policy_name}"
            
            self.iam.attach_role_policy(
                RoleName=role_name,
                PolicyArn=policy_arn
            )
            return response['Role']['Arn']
        except Exception as e:
            logger.error(f"Error creating role: {str(e)}")
            raise
    def create_lambda_package(self, lambda_dir=None):
        """
        创建Lambda部署包
        Args:
            lambda_dir: Lambda函数源代码目录，默认为alarm/lambda/aurora_slowlog_processor
        Returns:
            bytes: ZIP文件内容
        """
        try:
            # 创建临时目录
            temp_dir = tempfile.mkdtemp()
            zip_path = os.path.join(temp_dir, "lambda_package.zip")
            
            # 获取Lambda函数源代码路径
            if not lambda_dir:
                # 尝试从当前文件路径推断Lambda目录
                current_file = os.path.abspath(__file__)
                blue_script_dir = current_file.split('alarm')[0] + 'alarm'
                lambda_dir = os.path.join(
                    blue_script_dir,
                    'lambda',
                    'aurora_slowlog_processor'
                )
            
            if not os.path.exists(lambda_dir):
                raise FileNotFoundError(f"Lambda directory not found: {lambda_dir}")
                
            logger.info(f"Creating Lambda package from directory: {lambda_dir}")
            
            # 创建ZIP文件
            with zipfile.ZipFile(zip_path, 'w', zipfile.ZIP_DEFLATED) as zipf:
                for root, _, files in os.walk(lambda_dir):
                    for file in files:
                        file_path = os.path.join(root, file)
                        arcname = os.path.relpath(file_path, lambda_dir)
                        zipf.write(file_path, arcname)
            
            # 读取ZIP文件内容
            with open(zip_path, 'rb') as f:
                zip_content = f.read()
            
            # 清理临时目录
            shutil.rmtree(temp_dir)
            
            return zip_content
        except Exception as e:
            logger.error(f"Error creating Lambda package: {str(e)}")
            raise
            
    def create_or_update_lambda(self, function_name=None, role_arn=None, zip_content=None):
        """
        创建或更新Lambda函数
        Args:
            function_name: Lambda函数名称，默认为DEFAULT_LAMBDA_FUNCTION_NAME
            role_arn: Lambda执行角色ARN，如果未提供则自动创建
            zip_content: Lambda部署包内容，如果未提供则自动创建
        Returns:
            str: Lambda函数ARN
        """
        function_name = function_name or self.DEFAULT_LAMBDA_FUNCTION_NAME
        
        try:
            # 如果未提供角色ARN，则创建角色
            if not role_arn:
                role_arn = self.create_lambda_role()
                
            # 如果未提供部署包内容，则创建部署包
            if not zip_content:
                zip_content = self.create_lambda_package()
            
            # 设置环境变量
            environment_variables = {
                'Variables': {
                    'SLOW_QUERY_TIME': '1.0',  # 默认1秒
                    'NO_INDEX_RATIO': '100.0'  # 默认100
                }
            }
            # 检查函数是否已存在
            try:
                self.lambda_client.get_function(FunctionName=function_name)
                # 更新函数代码
                response = self.lambda_client.update_function_code(
                    FunctionName=function_name,
                    ZipFile=zip_content,
                    Publish=False
                )
                # 更新环境变量
                self.lambda_client.update_function_configuration(
                    FunctionName=function_name,
                    Environment=environment_variables
                )
                logger.info(f"Updated Lambda function: {function_name}")
            except self.lambda_client.exceptions.ResourceNotFoundException:
                # 创建函数
                response = self.lambda_client.create_function(
                    FunctionName=function_name,
                    Runtime='python3.9',
                    Role=role_arn,
                    Handler='lambda_function.lambda_handler',
                    Code={
                        'ZipFile': zip_content
                    },
                    Description='Aurora SlowLog Processor',
                    Timeout=60,
                    MemorySize=128,
                    Publish=True,
                    Environment=environment_variables
                )
                logger.info(f"Created Lambda function: {function_name}")
            lambda_arn=response['FunctionArn']
             # 添加Lambda权限 - 确保CloudWatch Logs可以调用Lambda
            self.add_lambda_permission(lambda_arn=lambda_arn, function_name=function_name)
            # 确保Lambda函数已经准备好接收调用
            logger.info(f"Waiting for Lambda function to be fully ready...")
            time.sleep(15)
            return lambda_arn
        except Exception as e:
            logger.error(f"Error creating/updating Lambda function: {str(e)}")
            raise
        
    def create_log_subscription(self, cluster, lambda_arn):
        """
        为Aurora集群的慢查询日志组创建订阅筛选器
        Args:
            cluster: Aurora集群信息
            lambda_arn: Lambda函数ARN
        """
        try:    
            # 从集群信息获取日志组名称
            cluster_id = cluster['DBClusterIdentifier']
            log_group_name = f"/aws/rds/cluster/{cluster_id}/slowquery"
            
            # 检查日志组是否存在
            
            log_groups = self.cloudwatch.describe_log_groups(logGroupNamePrefix=log_group_name)
            if not any(log_group['logGroupName'] == log_group_name for log_group in log_groups.get('logGroups', [])):
                # 日志组不存在，创建它
                logger.info(f"Log group does not exist: {log_group_name}")
                self.cloudwatch.create_log_group(logGroupName=log_group_name)
                logger.info(f"Log group created: {log_group_name}")
            else:
                logger.info(f"Log group already exists: {log_group_name}")
           
            
            filter_name = f"AuroraSlowLogProcessor"
            # 检查是否已存在订阅筛选器
            existing_filters = self.cloudwatch.describe_subscription_filters(
                logGroupName=log_group_name,
                filterNamePrefix=filter_name
            )
            
            if not existing_filters.get('subscriptionFilters'):
                # 创建新的订阅筛选器
                logger.info(f"Creating subscription filter for log group: {log_group_name}")
                self.cloudwatch.put_subscription_filter(
                    logGroupName=log_group_name,
                    filterName=filter_name,
                    filterPattern='',  # 处理所有日志事件
                    destinationArn=lambda_arn
                )
                logger.info(f"Successfully created subscription filter for log group: {log_group_name}")
            else:
                logger.info(f"Subscription filter already exists for log group: {log_group_name}")
        except Exception as e:
            logger.error(f"Error creating log subscription: {str(e)}")
            raise e
            
    def add_lambda_permission(self, lambda_arn=None, function_name=None):
        """
        添加Lambda权限，允许CloudWatch Logs调用函数
        Args:
            lambda_arn: Lambda函数ARN
            function_name: Lambda函数名称，如果未提供lambda_arn则使用此名称
        """
        try:
            account_id = self._get_account_id()
            region = self.region
            
            # 清理现有权限
            statement_ids = ['AllowCloudWatchLogsRDS']
            for statement_id in statement_ids:
                try:
                    self.lambda_client.remove_permission(
                        FunctionName=function_name,
                        StatementId=statement_id
                    )
                    logger.info(f"Removed existing permission: {statement_id}")
                except Exception:
                    pass
            
            # 添加权限 - 使用正确的SourceArn格式
            self.lambda_client.add_permission(
                FunctionName=function_name,
                StatementId='AllowCloudWatchLogsRDS',
                Action='lambda:InvokeFunction',
                Principal='logs.amazonaws.com',
                SourceArn=f"arn:aws:logs:{region}:{account_id}:*"
            )
            
            logger.info(f"Added CloudWatch Logs permission with SourceArn for Lambda function: {function_name}")
            
            # 验证权限是否正确添加
            try:
                policy_response = self.lambda_client.get_policy(FunctionName=function_name)
                logger.info(f"Lambda function policy: {policy_response['Policy']}")
            except Exception as e:
                logger.warning(f"Could not retrieve Lambda policy for verification: {str(e)}")
            
        except Exception as e:
            logger.error(f"Error adding Lambda permission: {str(e)}")
            raise

    def _scan_aurora_clusters_by_pattern(self, func, pattern):
        """
        根据模式遍历Aurora集群
        Args:
            func: 设置函数
            pattern: Aurora集群ID匹配模式
        """
        logger.info(f"Searching for Aurora clusters matching pattern: {pattern}")
        found = 0
        paginator = self.rds.get_paginator('describe_db_clusters')
        
        for page in paginator.paginate():
            for cluster in page['DBClusters']:
                # 只处理Aurora MySQL集群
                if 'Engine' in cluster and 'aurora-mysql' in cluster['Engine'].lower():
                    if re.match(pattern, cluster['DBClusterIdentifier']):
                        logger.info(f"Found matching Aurora cluster: {cluster['DBClusterIdentifier']}")
                        func(cluster)
                        found += 1
        
        logger.info(f"Found {found} Aurora cluster(s) matching pattern: {pattern}" if found else f"No Aurora instances found matching pattern: {pattern}")