import json
import boto3
import re
import os
import logging
from datetime import datetime

# 配置日志
logger = logging.getLogger()
logger.setLevel(logging.INFO)

# 指标命名空间和名称
METRIC_NAMESPACE = 'RDS/AuroraSlowLog'
METRIC_NAME_SLOW_QUERY = 'AuroraSlowQueryCount'
METRIC_NAME_NO_INDEX = 'AuroraNoIndexSlowQueryCount'

# 环境变量配置
SLOW_QUERY_TIME = float(os.environ.get('SLOW_QUERY_TIME', '1.0'))  # 默认1秒
NO_INDEX_RATIO = float(os.environ.get('NO_INDEX_RATIO', '100.0'))   # 默认100
"""
| Ratio Range | Interpretation |
|-------------|----------------|
| 1-2         | Excellent - Nearly optimal index usage |
| 3-10        | Good - Reasonable index usage |
| 10-100      | Concerning - Potential index issues |
| 100+        | Poor - Likely missing or ineffective indexes |
"""
def lambda_handler(event, context):
    """
    处理Aurora慢查询日志并生成CloudWatch指标
    """
    logger.info(f"Processing event: {json.dumps(event)}")
    
    # 初始化CloudWatch客户端
    cloudwatch = boto3.client('cloudwatch')
    
    # 处理日志事件
    if 'awslogs' in event:
        # 解析CloudWatch Logs事件
        try:
            import base64
            import gzip
            from io import BytesIO
            
            # 解码和解压缩日志数据
            compressed_payload = base64.b64decode(event['awslogs']['data'])
            uncompressed_payload = gzip.GzipFile(fileobj=BytesIO(compressed_payload)).read()
            payload = json.loads(uncompressed_payload)
            
            logger.info(f"Decoded payload: {json.dumps(payload)}")
            
            # 提取日志组名称和日志流名称
            log_group = payload['logGroup']
            log_stream = payload['logStream']
            
            # 从日志组名称中提取集群ID
            # 格式: /aws/rds/cluster/{cluster_id}/slowquery
            cluster_id = None
            match = re.search(r'/aws/rds/cluster/([^/]+)/slowquery', log_group)
            if match:
                cluster_id = match.group(1)
            
            if not cluster_id:
                logger.warning(f"Could not extract cluster ID from log group: {log_group}")
                return {
                    'statusCode': 400,
                    'body': json.dumps('Could not extract cluster ID from log group')
                }
            
            # 处理日志事件
            slow_query_count = 0
            no_index_query_count = 0
            
            for log_event in payload['logEvents']:
                message = log_event.get('message', '')
                
                # 提取时间戳
                timestamp = None
                time_match = re.search(r'# Time:\s+(\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d+Z)', message)
                if time_match:
                    time_str = time_match.group(1)
                    try:
                        timestamp = datetime.fromisoformat(time_str.replace('Z', '+00:00'))
                    except ValueError:
                        logger.warning(f"Could not parse timestamp: {time_str}")
                    
                # 检查是否为慢查询
                query_time_match = re.search(r'Query_time:\s+(\d+\.\d+)', message)
                if query_time_match:
                    query_time = float(query_time_match.group(1))
                    
                    # 只有当查询时间超过阈值时才记录
                    if query_time > SLOW_QUERY_TIME:
                        slow_query_count += 1
            
                    # 提取Rows_examined和Rows_sent
                    rows_examined_match = re.search(r'Rows_examined:\s+(\d+)', message)
                    rows_sent_match = re.search(r'Rows_sent:\s+(\d+)', message)
                    
                    if rows_examined_match and rows_sent_match:
                        rows_examined = int(rows_examined_match.group(1))
                        rows_sent = int(rows_sent_match.group(1))
                        # 计算比率并与阈值比较
                        if rows_examined > 0 and rows_sent > 0:
                            no_index_ratio = rows_examined / rows_sent
                            if no_index_ratio > NO_INDEX_RATIO:
                                no_index_query_count += 1
            
            # 发布指标
            if slow_query_count > 0:
                metric_data = {
                    'MetricName': METRIC_NAME_SLOW_QUERY,
                    'Dimensions': [
                        {
                            'Name': 'DBClusterIdentifier',
                            'Value': cluster_id
                                }
                            ],
                            'Value': slow_query_count,
                            'Unit': 'Count'
                        }
                
                # 如果有时间戳，添加到指标数据中
                if timestamp:
                    metric_data['Timestamp'] = timestamp
                cloudwatch.put_metric_data(
                    Namespace=METRIC_NAMESPACE,
                    MetricData=[metric_data]
                )
                logger.info(f"Published {slow_query_count} slow query metrics for cluster {cluster_id}")
            
            if no_index_query_count > 0:
                metric_data = {
                            'MetricName': METRIC_NAME_NO_INDEX,
                            'Dimensions': [
                                {
                                    'Name': 'DBClusterIdentifier',
                                    'Value': cluster_id
                                }
                            ],
                            'Value': no_index_query_count,
                            'Unit': 'Count'
                        }
                
                # 如果有时间戳，添加到指标数据中
                if timestamp:
                    metric_data['Timestamp'] = timestamp
                
                cloudwatch.put_metric_data(
                    Namespace=METRIC_NAMESPACE,
                    MetricData=[metric_data]
                )
                logger.info(f"Published {no_index_query_count} no-index slow query metrics for cluster {cluster_id}")
            
            return {
                'statusCode': 200,
                'body': json.dumps(f'Processed {slow_query_count} slow queries, {no_index_query_count} no-index queries')
            }
            
        except Exception as e:
            logger.error(f"Error processing log event: {str(e)}")
            return {
                'statusCode': 500,
                'body': json.dumps(f'Error: {str(e)}')
            }
    
    return {
        'statusCode': 400,
        'body': json.dumps('Invalid event format')
    }